/**
 * @file    advance_software_protect_port.c
 * @brief   ICWorkShop Universal MCU Anti-Debugger library
 * @author  csh@icworkshop.com iphone (+86) 15989373832
 *
 * Safety License Shield
 * Copyright (c) 2017-2023, ICWorkshop Limited, All Rights Reserved
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *-------------------------------------------------------------------------------------
 *The following codes are user-defined codes. The purpose is to reduce the
 *probability of overlap between the encryption and verification algorithms of
 *each project, and to remove the universal characteristics of the universal
 *verification algorithms, such as CRC32, CRC16, AES, RSA and other encryption
 *algorithms
 *-------------------------------------------------------------------------------------
 */

/* STM32F103xG sample */
#ifdef STM32F103xG
#include "stm32f1xx_hal.h"
#endif

#include "advance_software_protect_user.h"

/*
 *	What is demonstrated here is a standard CRC, with your own CRC function
 *substituted in a real project, in order to avoid some common features and make
 *the analysis more difficult for the reverse analyst for object watch module
 */
ASP_ENCRYPT_F_DECL(uint32_t advance_software_protect_crc32(uint8_t *buffer,
														   size_t size))
uint32_t advance_software_protect_crc32(uint8_t *buffer, size_t size)
{
	uint32_t crc = crc32_inner(buffer, size);
	return crc;
}
/* for signature */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE
/*
 * @brief   Gets the chip ID of the current chip.
 * @pram 	  id_data	: A buffer that stores ids
 *					size	: id length
 * @retval void
 */
ASP_ENCRYPT_F_DECL(void advance_software_protect_get_cid(uint32_t *id_addr, uint8_t size,uint32_t *id_data))
void advance_software_protect_get_cid(uint32_t *id_addr, uint8_t size, uint32_t *id_data)
{
	// If the ID is discontinuous, you need to handle it yourself...
	// offset 0x00,0x04,0x14 ( Such as STM32L0, STM32L1, STM32L4 etc)
#if ((defined STM32L0) || (defined STM32L1) || (defined STM32L4))
	id_data[0] = *(id_addr + (0x00 / 4));
	id_data[1] = *(id_addr + (0x04 / 4));
	id_data[2] = *(id_addr + (0x14 / 4));
#else
	// The ID is continuous...
	for (int i = 0; i < (size / 4); i++){
		*id_data++ = *id_addr++;
	}
#endif
}
#endif

/*
 * @brief   Reset target system
 * @pram 	  none
 * @retval	bool
 */
ASP_ENCRYPT_F_DECL(void advance_software_protect_system_reset())
void advance_software_protect_system_reset()
{
#ifdef RELEASE
	NVIC_SystemReset();
#endif
}

/* for anti debugger*/
#ifdef ADVANCE_SOFTWARE_PROTECT_ANTI_DEBUGGER_ENABLE
/*
 * @brief   Disabling the debugger
 * @pram 	  none
 * @retval	none
 */
ASP_ENCRYPT_F_DECL(void advance_software_protect_debugger_disable())
void advance_software_protect_debugger_disable()
{
#ifdef RELEASE
	/* deiable debugger */
	__HAL_AFIO_REMAP_SWJ_DISABLE();
#endif
}
/*
 * @brief   Check whether read protection is enabled
 * @pram 	  none
 * @retval	true is enable rdp
 */
ASP_ENCRYPT_F_DECL(bool advance_software_protect_rdp_check())
bool advance_software_protect_rdp_check()
{
#ifdef RELEASE
	/*check RDP */
	FLASH_OBProgramInitTypeDef m_cur_ob;
	HAL_FLASHEx_OBGetConfig(&m_cur_ob);
	return (m_cur_ob.RDPLevel != OB_RDP_LEVEL_0);
#else
	return true; /* default return true */
#endif
}
/*
 * @brief   Disabling Read Protection
 * @pram 	  none
 * @retval	none
 */
ASP_ENCRYPT_F_DECL(void advance_software_protect_rdp_enable())
void advance_software_protect_rdp_enable()
{
#ifdef RELEASE
	bool rdp_result = {0};
	/* get current ob */
	FLASH_OBProgramInitTypeDef m_cur_ob;
	HAL_FLASHEx_OBGetConfig(&m_cur_ob);

	/* unlock ob */
	if (HAL_FLASH_Unlock() == HAL_OK && HAL_FLASH_OB_Unlock() == HAL_OK)
	{

		/* Set RDP to Level 1 or Level 2*/
		m_cur_ob.RDPLevel = OB_RDP_LEVEL_1;

		/* update ob */
		rdp_result = (HAL_FLASHEx_OBProgram(&m_cur_ob) == HAL_OK);

		/* lock flash */
		HAL_FLASH_OB_Lock();
		HAL_FLASH_Lock();

		/*
			To make the option byte take effect, note that by resetting the target chip,
			the target chip reloads the option byte to make it take effect
		*/
		if (rdp_result)
		{
			HAL_FLASH_OB_Launch();
		}
	}
#endif
}
#endif

#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE
/* for advance software protect matrix */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_GCC
#pragma GCC diagnostic ignored "-Wparentheses"
#endif
ASP_ENCRYPT_F_DECL(static void ChipUIDAlgo(char pUserID[], char pChipID[], char pKey[]))
	
//The following code may warn in KEIL(MDK), ignore it
static void ChipUIDAlgo(char pUserID[], char pChipID[], char pKey[])
{
	pKey[0] = pUserID[8] | pChipID[5] - pUserID[10] + pChipID[2] ;
	pKey[1] = pChipID[1] & pChipID[0] ^ pUserID[4] * pChipID[11] ;
	pKey[2] = pChipID[3] ^ pChipID[10] | pChipID[6] - pUserID[11] ;
	pKey[3] = pUserID[9] & pChipID[4] + pUserID[6] * pUserID[0] ;
	pKey[4] = pChipID[7] + pUserID[1] * pChipID[9] - pUserID[7] ;
	pKey[5] = pUserID[2] & pUserID[3] ^ pUserID[5] | pChipID[8] ;
	pKey[6] = pChipID[6] * pChipID[4] & pUserID[6] + pChipID[11] ;
	pKey[7] = pChipID[1] ^ pUserID[0] - pUserID[5] | pUserID[7] ;
	pKey[8] = pChipID[8] - pUserID[10] | pChipID[2] * pUserID[2] ;
	pKey[9] = pChipID[10] & pUserID[11] + pChipID[5] ^ pUserID[4] ;
	pKey[10] = pUserID[9] & pUserID[3] ^ pChipID[7] - pChipID[0] ;
	pKey[11] = pChipID[3] + pUserID[8] | pUserID[1] * pChipID[9] ;
}

#if UID_KEYADDR_PLACEHOLDER_EN
/* for IAR */
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR
#pragma diag_suppress = Pa082
const static char m_asp_signature_license_storage[UID_KEY_LENGTH] @UID_KEYADDR_INNER_VAL;
#else
/* for mdk & gcc*/
const static char m_asp_signature_license_storage[UID_KEY_LENGTH] __attribute__((section(UID_KEYADDR_INNER))); /* key place holder */
#endif
#endif
/* for advance software protect ecdsa */
#else
#if UID_KEYADDR_PLACEHOLDER_EN
/* for IAR */
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR
#pragma diag_suppress = Pa082
const static S_ECDSA_SIGN m_asp_signature_license_storage @UID_KEYADDR_INNER_VAL;
#else
/* for mdk & gcc*/
const static S_ECDSA_SIGN m_asp_signature_license_storage __attribute__((section(UID_KEYADDR_INNER))); /* key place holder */
#endif
#endif

#endif
#endif

/* skip sn */
#ifdef ADVANCE_SOFTWARE_PROTECT_SN_ENABLE
extern uint32_t m_serialnumber;
#endif

/* signature config */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE

const E_ADV_SOFTWARE_PROTECT_SIGNATURE m_signature_config = {
	/* define rand factor to obfuscate data */
	.m_rand_factor = ASP_RAND_FACTOR,
	/* chip id information */
	.m_chipid_addr = ASP_SIGNATURE_OBFUSCATE(UID_CHIP_ADDR),
	.m_chipid_size = ASP_SIGNATURE_OBFUSCATE(UID_CHIP_SIZE),
	/*
	 * Setting the read chip ID callback, set to NULL means using
	 * advance_software_protect_get_cid instead, refer to
	 * "void advance_software_protect_get_cid(uint32_t * id_addr, uint8_t size,uint32_t *id_data)"
	 */
	.m_chipid_usr_cb = advance_software_protect_get_cid,
/* signature type */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX
	.m_type = SignatureMatrix,
	.m_signature_data.m_matrix.m_uid_key_length = ASP_SIGNATURE_OBFUSCATE(UID_KEY_LENGTH),
	.m_signature_data.m_matrix.m_uid_key_addr = ASP_SIGNATURE_OBFUSCATE(UID_KEYADDR_INNER_VAL),
	.m_signature_data.m_matrix.m_user_id_key[0] = UID_USERID_KEY1,
	.m_signature_data.m_matrix.m_user_id_key[1] = UID_USERID_KEY2,
	.m_signature_data.m_matrix.m_user_id_key[2] = UID_USERID_KEY3,
	.m_signature_data.m_matrix.m_endian = UID_DATAENDIAN,
	.m_signature_data.m_matrix.m_calc_callback = ChipUIDAlgo,
#else
	.m_type = SignatureECDSA,
	.m_signature_data.m_ecdsa.m_ecdsa_public_key = PUBLIC_KEY,
	.m_signature_data.m_ecdsa.m_ecdsa_license_addr = (uint32_t *)UID_KEYADDR_INNER_VAL,
#endif
};
#endif

S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_START
/* Please add the dynamic storage area you want to skip, refer to the sequence
 * number definition below and add the definition after it.
 */
/***************************** START *********************************/
/* skip sn */
#ifdef ADVANCE_SOFTWARE_PROTECT_SN_ENABLE
	{
		.m_skip_addr = (void *)&m_serialnumber,
		.m_skip_size = sizeof(m_serialnumber)},
#endif
	/* Add items that you need to skip validation */
	/* ... */
	/****************************** END **********************************/
	S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_END S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_APP
