/**
 * @file    advance_software_protect_samples.c
 * @brief   ICWorkShop Universal MCU Anti-Debugger library
 * @author  csh@icworkshop.com iphone (+86) 15989373832
 *
 * Safety License Shield
 * Copyright (c) 2017-2023, ICWorkshop Limited, All Rights Reserved
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifdef STM32F103xG
#include "stm32f1xx_hal.h"
#endif
#include "advance_software_protect_user.h"

/* signature in porting file */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE
extern const E_ADV_SOFTWARE_PROTECT_SIGNATURE m_signature_config;
#endif

/* message callback */
#ifdef STM32F103xG
extern UART_HandleTypeDef huart1;
#endif
static void advance_software_protect_log_output(const char *msg)
{
#ifdef STM32F103xG
    HAL_UART_Transmit(&huart1, (uint8_t *)msg, strlen(msg), strlen(msg));
#endif
}

/* memory alloc & free */
void *calloc_mem_func(size_t count, size_t size)
{
    return calloc(count, size);
}
void free_mem_func(void *blk_obj)
{
    free(blk_obj);
}

/* define product serial number in flash */
#ifdef ADVANCE_SOFTWARE_PROTECT_SN_ENABLE
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_GCC
#define ADVANCE_SOFTWARE_PROTECT_SN_ADDR ".advance_software_protect_sn" /*defined in ld link Script */
#elif defined(ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR)
#define ADVANCE_SOFTWARE_PROTECT_SN_ADDR 0x08001000
#else
#define ADVANCE_SOFTWARE_PROTECT_SN_ADDR ".ARM.__at_0x08000700"
#endif
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR
#pragma diag_suppress = Pa082
const uint32_t m_serialnumber @ADVANCE_SOFTWARE_PROTECT_SN_ADDR = {0}; /* Fixed address for easy maintenance */
#else
const uint32_t m_serialnumber __attribute__((section(ADVANCE_SOFTWARE_PROTECT_SN_ADDR))) = {0}; /* Fixed address for easy maintenance */
#endif
#endif

/*
 *	The following test cases describe how to use the UNIVERSAL advanced software protection algorithm of MCU provided by Power Writer.
 * Please refer to the corresponding description or documentation for different functions
 */

/* Simple Object define for Object watcher */
typedef struct S_Object
{
    uint32_t m_sample1;
    uint32_t m_sample2;
    uint32_t m_sample3;
    uint8_t *m_ptr8;
} S_Object;

typedef struct S_A
{
    S_Object m_data;
    int m_count;
} S_A;

typedef struct S_B
{
    S_A m_packer;
} S_B;

typedef struct S_Class
{
    uint32_t m_obj_A;
    uint32_t m_obj_B;
    uint32_t m_obj_C;
    union
    {
        S_A m_a;
        S_B m_b;
    } _data_;
} S_Class;

/* Wrapper S_Object object structure */
S_ADV_SOFTWARE_PROTECT_OBJECTWATCH_Watcher(S_Object)

    /* Single object monitoring */
    S_Object_Watcher m_Object;

/*
 *		Single Object Watcher
 */
void advance_software_protect_single_object_watcher()
{
    advance_software_protect_logger(ADV_DEBUG, " ---------------Single Object Watcher Simple Start ----------\r\n");
    /* first time will be passed */
    bool m_object_check_result = advance_software_protect_objectwatch_check(&m_Object, S_Object_Watcher);

    /* print result */
    if (m_object_check_result)
    {
        advance_software_protect_logger(ADV_DEBUG, " m_Object check passed at first time\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_ERROR, " m_Object check failed at first time, Never execute at first time!\r\n");
    }

    /* Emulation external modifies memory, but does not update verification information, and will fail to check ! */
    m_Object.m_Object.m_sample1 = 1;
    m_Object.m_Object.m_sample2 = 2;
    m_Object.m_Object.m_sample3 = 3;
    m_Object.m_Object.m_ptr8 = (uint8_t *)0x12345678;

    m_object_check_result = advance_software_protect_objectwatch_check(&m_Object, S_Object_Watcher);

    /* print result */
    if (m_object_check_result)
    {
        advance_software_protect_logger(ADV_DEBUG, " m_Object has been modified but there is no update validation, which is not performed here\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_ERROR, " m_Object will fail if it is modified without updating the checksum value!\r\n");
    }

    /* It will pass after the checksum is updated  */
    advance_software_protect_objectwatch_update(&m_Object, S_Object_Watcher); /* update checksum */
    m_object_check_result = advance_software_protect_objectwatch_check(&m_Object, S_Object_Watcher);
    /* print result */
    if (m_object_check_result)
    {
        advance_software_protect_logger(ADV_DEBUG, " m_Object had modified & updated, will be passed !\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_ERROR, " m_Object had modified & updated, never failed !\r\n");
    }
    advance_software_protect_logger(ADV_DEBUG, " ---------------Single Object Watcher Simple End----------\r\n");
}

/*
 *		Dynamic Muti Object Watcher
 */
void advance_software_protect_dynamic_muti_object_watcher_callback(E_ObjectWatchException excep, void *param)
{
    switch (excep)
    {
    /* Parameter error */
    case ObjectParamError:
        advance_software_protect_logger(ADV_WARN, " Error param ...\r\n");
        break;
    /* The object is modified */
    case ObjectModified:
        advance_software_protect_logger(ADV_ERROR, " [0x%08X] Object was modified!\r\n", (unsigned int)(uint32_t)param);
        break;
    /* The object already exits */
    case ObjectAlreadyExits:
        advance_software_protect_logger(ADV_WARN, " [0x%08X] Object already exist!\r\n", (unsigned int)(uint32_t)param);
        break;
    /* Not enough memory */
    case ObjectMemoryError:
        advance_software_protect_logger(ADV_ERROR, " Memory space is used up!\r\n");
        break;
    }
}

/* Test Object for dynamic object watcher	*/
S_Object m_object;
S_A m_aType;
S_B m_bType;
S_Class m_class;

void advance_software_protect_dynamic_muti_object_watcher()
{
    advance_software_protect_logger(ADV_DEBUG, " ---------- Dynamic Muti-Object Watcher Simple Start ----------\r\n");
    /* set event callback */
    advance_software_protect_logger(ADV_DEBUG, " Set ObjectWatcher Event callback...\r\n");
    advance_software_protect_objectwatch_list_set_callback(advance_software_protect_dynamic_muti_object_watcher_callback);

    /* add object to watcher */
    advance_software_protect_objectwatch_list_add(&m_object, sizeof(m_object), false);
    advance_software_protect_objectwatch_list_add(&m_aType, sizeof(m_aType), false);
    advance_software_protect_objectwatch_list_add(&m_bType, sizeof(m_bType), false);
    advance_software_protect_objectwatch_list_add(&m_class, sizeof(m_class), false);

    advance_software_protect_objectwatch_list_add(&m_class, sizeof(m_class), false); // Raises that the object already has an exception (ObjectAlreadyExits)
    advance_software_protect_objectwatch_list_add(0, sizeof(m_class), false);        // A parameter error is triggered (ObjectParamError)

    // Dynamically add managed objects
    // AdvanceSoftwareProtectLogger(ADV_INFO, " Alloc 100 dynamic object..\r\n");
		do{
		    S_Class *m_new_obj = (S_Class *)calloc(1, sizeof(S_Class));
        if (m_new_obj){
            advance_software_protect_objectwatch_list_add(m_new_obj, sizeof(S_Class), true); // When you remove an object, you also delete the object
        }
        else{
            // memory error
            break;
        }
		}while(true);

    advance_software_protect_logger(ADV_INFO, " Alloc 100 dynamic object done ..\r\n");
    /* Find object */
    bool m_find = advance_software_protect_objectwatch_list_find(&m_aType);
    if (m_find)
    {
        advance_software_protect_logger(ADV_INFO, " m_aType exist..\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_INFO, " m_aType not exist..\r\n");
    }

    /* count of watch objects */
    advance_software_protect_logger(ADV_DEBUG, " Total %d objects in watching..\r\n", advance_software_protect_objectwatch_list_size());

    /* check single object */
    bool m_check = advance_software_protect_objectwatch_list_check(&m_class);
    if (m_check)
    {
        advance_software_protect_logger(ADV_DEBUG, " m_class check passed ..\r\n");
    }
    else
    {
        // failed...
        //...
    }

    /* check all objects */
    m_class._data_.m_a.m_data.m_sample1 ^= 0xAB; // Modify the properties of an object

    size_t m_passed_count = advance_software_protect_objectwatch_list_check_all();
    size_t m_total_count = advance_software_protect_objectwatch_list_size();

    if (m_passed_count == m_total_count)
    {
        advance_software_protect_logger(ADV_DEBUG, " All objects check passed ..\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_DEBUG, " Passed: %d, Total : %d ..\r\n", m_passed_count, m_total_count);
    }

    /* update single object */
    if (advance_software_protect_objectwatch_list_update(&m_class))
    {
        advance_software_protect_logger(ADV_DEBUG, " m_class update ok ..\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_DEBUG, " m_class update failed ..\r\n");
    }

    advance_software_protect_objectwatch_list_check_all();

    /* delete object */
    bool m_del = advance_software_protect_objectwatch_list_remove(&m_bType);

    if (m_del)
    {
        advance_software_protect_logger(ADV_INFO, " m_bType removed, left %d object in watching..\r\n", advance_software_protect_objectwatch_list_size());
    }
    else
    {
        advance_software_protect_logger(ADV_INFO, " m_bType not exist!..\r\n");
    }

    /* update all object */
    advance_software_protect_objectwatch_list_update_all();
    advance_software_protect_logger(ADV_INFO, " All objects updated...\r\n");

    /* clear all objects */
    advance_software_protect_objectwatch_list_reset();
    advance_software_protect_logger(ADV_INFO, " All objects removed, left %d object in watching..\r\n", advance_software_protect_objectwatch_list_size());
    advance_software_protect_logger(ADV_DEBUG, " ---------------Dynamic Muti-Object Watcher Simple End----------\r\n");
}

void advance_software_protect_show_serialnumber()
{
    volatile uint32_t m_sn_addr = (uint32_t)&m_serialnumber;
    volatile uint32_t m_sn_data = *((uint32_t *)(m_sn_addr));

    advance_software_protect_logger(ADV_INFO, "---------------Product Serial Number Info Start----------\r\n");
    advance_software_protect_logger(ADV_INFO, " Product Serial number is : 0x%08X , In Flash Address 0x%08X.\r\n", (unsigned int)(m_sn_data), (unsigned int)m_sn_addr);
    advance_software_protect_logger(ADV_INFO, "---------------Product Serial Number Info End----------\r\n");
}

/* for signature */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE

void advance_software_protect_signature()
{
    advance_software_protect_logger(ADV_DEBUG, " ---------------Product Signature start----------\r\n");
    if (advance_software_protect_verify_authorization(&m_signature_config))
    {
        advance_software_protect_logger(ADV_DEBUG, " Chip signature verify passed...\r\n");
    }
    else
    {
        advance_software_protect_logger(ADV_ERROR, " Chip signature verify failed!!!!!\r\n");
    }
    advance_software_protect_logger(ADV_DEBUG, " ---------------Product Signature end----------\r\n");
		
		
}
#endif

/* for flash verify */
#ifdef ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_ENABLE
void advance_software_protect_flash_verify_sample()
{
    advance_software_protect_logger(ADV_DEBUG, " ---------------Flash Verify start----------\r\n");
    E_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_RESULT result = advance_software_protect_flash_verify(ADV_DATA_ENCRYPT_KEY);

    switch (result)
    {
    case ConfigError: /* configure error ! */
        advance_software_protect_logger(ADV_DEBUG, " Configure error !\r\n");
        break;

    case UnprocessedAfterComplied: /* Run in Debug mode */
        advance_software_protect_logger(ADV_DEBUG, " (Running in debugging mode) Flash Verify...\r\n");
        break;

    case FlashVerifyFailed: /* Verify Failed */
        advance_software_protect_logger(ADV_DEBUG, " Flash Verify failed!!!!\r\n");
        break;

    case FlashVerifyPassed: /* Verify Passed */
        advance_software_protect_logger(ADV_DEBUG, " Flash Verify passed...\r\n");
        break;
    }

    advance_software_protect_logger(ADV_DEBUG, " ---------------Flash Verify end----------\r\n");
}
#endif

/* for anti debugger */
#ifdef ADVANCE_SOFTWARE_PROTECT_ANTI_DEBUGGER_ENABLE
bool advance_software_protect_anti_debugger_sample(E_AntiDebuggerEvent event)
{
    switch (event)
    {
    /* Development mode running	*/
    case Development:
        advance_software_protect_logger(ADV_DEBUG, " (Running in development mode) Anti Debugger ...\r\n");
        return true;

    /* The RDP mode is disabled	*/
    case RDPDisabled:
        advance_software_protect_logger(ADV_DEBUG, " (Read protection is not enabled) Anti Debugger ...\r\n");
        return true; /* Return true if you want to enable RDP */

    /* The RDP mode is enabled  */
    case RDPEnabled:
        advance_software_protect_logger(ADV_DEBUG, " (Read protection enabled) Anti Debugger ...\r\n");
        return true; /* Return true if you want to reset */

    /* The system is resetting	*/
    case SystemReset:
        advance_software_protect_logger(ADV_DEBUG, " (Request system reset) Anti Debugger ...\r\n");
        return true; /* Return true if you want to reset */

    /* Disable Debugger	*/
    case DebuggerDisable:
        advance_software_protect_logger(ADV_DEBUG, " (Disabling the debugger) Anti Debugger ...\r\n");
        return true; /* Return true if you want to disable the debugger */
    }
    return true;
}
#endif

/* for ram code loader */
void advance_software_protect_encrypt_code_sample(E_ENCRYPT_CODE_EVENT event)
{
    switch (event)
    {

    /* Development mode running	*/
    case EncryptCodeUnprocessed:
        advance_software_protect_logger(ADV_EXCEPTION, " (Not processed !) Please use PowerWriter to programming encrypted firmware before debugging. ...\r\n");
        while (1)
        {
        } /* suspended */

    /* Loading RAM code succeeded	*/
    case EncryptCodeOk:
        advance_software_protect_logger(ADV_DEBUG, " (Successful) Dynamic function encryption ...\r\n");
        break;

    /* Failed to load RAM code  */
    case EncryptCodeError:
        advance_software_protect_logger(ADV_ERROR, " (Failure !!!!) Dynamic function encryption ...\r\n");
        /* reset system */
        advance_software_protect_system_reset();

        break;
    }
}

/*
 *					Advance Software Protect Library Samples
 *						        cshsoft / csh@icworkshop.com
 */

void advance_software_protect_rights()
{
    static bool once_only = false;
    if (!once_only)
    {
        once_only = true;
        advance_software_protect_logger(ADV_BLANK, "---------------------------------------------------------------------------------\r\n");
        advance_software_protect_logger(ADV_BLANK, "\t\tAdvance software protect lib\r\n");
        advance_software_protect_logger(ADV_BLANK, "\r\n");
        advance_software_protect_logger(ADV_BLANK, "This project provides a set of universal MCU software encryption algorithm.\r\n");
        advance_software_protect_logger(ADV_BLANK, "The goal is simple, easy to use, easy to integrate, and does not require any\r\n");
        advance_software_protect_logger(ADV_BLANK, "additional hardwareThe following are the test cases of the project, please \r\n");
        advance_software_protect_logger(ADV_BLANK, "select the appropriate use method, some functions need to be used together\r\n");
        advance_software_protect_logger(ADV_BLANK, "with repack tool, if you have any questions, please contact us\r\n");
        advance_software_protect_logger(ADV_BLANK, "\r\n");
        advance_software_protect_logger(ADV_BLANK, "\r\n");
        advance_software_protect_logger(ADV_BLANK, "PowerWriter team\r\n");
        advance_software_protect_logger(ADV_BLANK, "Version:%s\r\n", advance_software_protect_version());
        advance_software_protect_logger(ADV_BLANK, "Author: csh@icworkshop.com(cshsoft)\r\n");
        advance_software_protect_logger(ADV_BLANK, "Site: www.powerwriter.com\r\n");
        advance_software_protect_logger(ADV_BLANK, "Docs: https://docs.powerwriter.com/docs/next/powerwriter_for_arm/application_note/advance_software_protect\r\n");
        advance_software_protect_logger(ADV_BLANK, "---------------------------------------------------------------------------------\r\n");
    }
}

/*
 * IMPORTANT:
 * Do not encrypt the parent function () that contains the
 * advance_software_protect_encryption_code_loader!
 */
void advance_software_protect_init()
{
    /* initial */
		advance_software_protect_set_mem_calloc_free_callback(calloc_mem_func, free_mem_func);
    advance_software_protect_set_logger_callback(advance_software_protect_log_output);
    /* show informations */
    advance_software_protect_rights();
#ifdef ADVANCE_SOFTWARE_PROTECT_CODE_ENCRYPT_ENABLE
    /* encrypt code loader	*/
    advance_software_protect_encryption_code_loader(ADV_ENCRYPT_CODE_SEGMENT_ADDR, ADV_DATA_ENCRYPT_KEY, advance_software_protect_encrypt_code_sample);
#endif
}

ASP_ENCRYPT_F_DECL(void advance_software_protect_sample())
void advance_software_protect_sample()
{
    /* show production serial number */
#ifdef ADVANCE_SOFTWARE_PROTECT_SN_ENABLE
    advance_software_protect_show_serialnumber();
#endif
    /* single object watcher */
    advance_software_protect_single_object_watcher();

    /* dynamic muti-object watcher */
    advance_software_protect_dynamic_muti_object_watcher();

    /* chip signature */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE
		advance_software_protect_signature();
#endif

    /* flash verify */
#ifdef ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_ENABLE
    advance_software_protect_flash_verify_sample();
#endif

#ifdef ADVANCE_SOFTWARE_PROTECT_ANTI_DEBUGGER_ENABLE
    /* anti debugger */
    advance_software_protect_anti_debugger(advance_software_protect_anti_debugger_sample);
#endif
}
