/**
 * @file    advance_software_protect_user.h
 * @brief   ICWorkShop Universal MCU Anti-Debugger library
 * @author  csh@icworkshop.com iphone (+86) 15989373832
 *
 * Safety License Shield
 * Copyright (c) 2017-2024, ICWorkshop Limited, All Rights Reserved
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __advance_software_protect_user_h
#define __advance_software_protect_user_h

#ifdef __cplusplus
extern "C"
{
#endif

#include "./advance_software_protect_lib.h"

/* Must be modified in real projects */
#define ASP_RAND_FACTOR (0x45623513)

/*  segment address */
#ifdef ADVANCE_SOFTWARE_PROTECT_CODE_ENCRYPT_ENABLE
#if (defined ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR) && (defined RELEASE)
#define ADV_ENCRYPT_CODE_SEGMENT_ADDR (0x20017004) /* refer to advance_software_protect_encryption_section + 4, see hex file */  
#else
#define ADV_ENCRYPT_CODE_SEGMENT_ADDR (0x20017000) /* refer to advance_software_protect_encryption_section */
#endif
#endif

/* key for data encrypt */
#if (defined ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_ENABLE) || (defined ADVANCE_SOFTWARE_PROTECT_CODE_ENCRYPT_ENABLE)
#define ADV_DATA_ENCRYPT_KEY (0xfb90ffcb9b62c2e6) /* 64 bit*/
#endif

/* Enable 4-Byte Serial Number */
#define ADVANCE_SOFTWARE_PROTECT_SN_ENABLE

/* Signature */
//#define ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX /* using matrix as signature */
#define ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ECDSA /* using ecdsa as signature */

#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE

/* Define random numbers to obfuscate key data and must be adjusted in real projects */
#define UID_CHIP_ADDR (0x1FFFF7E8)	   /* refer to (UID_CHIP_ADDR_MIX ^ UID_CHIP_ADDR) in cortex_chipid_binding.h */
#define UID_CHIP_SIZE (12)			   /* refer to UID_CHIP_SIZE in cortex_chipid_binding.h */
#define UID_KEYADDR_PLACEHOLDER_EN (1) /* Whether or not the certificate storage address is pre-assigned in the firmware */

#define UID_KEYADDR_INNER_VAL (0x08002000) /* refer to UID_KEYADDR_INNER in cortex_chipid_binding.h  */
#ifdef ADVANCE_SOFTWARE_PROTECT_COMPILER_MDK
#define UID_KEYADDR_INNER ".ARM.__at_0x08002000" /* refer to UID_KEYADDR_INNER in cortex_chipid_binding.h  */
#elif defined ADVANCE_SOFTWARE_PROTECT_COMPILER_GCC
#define UID_KEYADDR_INNER ".asp_signature_license" /* refer to UID_KEYADDR_INNER in cortex_chipid_binding.h  */
#endif

/* using matrix to signature */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX
#define UID_KEY_LENGTH (12)			 /* refer to UID_KEY_LENGTH in cortex_chipid_binding.h  */
#define UID_DATAENDIAN littleEndian	 /* refer to UID_DATAENDIAN in cortex_chipid_binding.h */
#define UID_USERID_KEY1 (0x53A886F5) /* refer to UserID 1 in cortex_chipid_binding.h */
#define UID_USERID_KEY2 (0x5D71DDB3) /* refer to UserID 2 in cortex_chipid_binding.h */
#define UID_USERID_KEY3 (0x32677FE8) /* refer to UserID 3 in cortex_chipid_binding.h */
#else
	/* using ecdsa to signature */
	// Public key
	const static uint8_t PUBLIC_KEY[49] = {
		0x04, 0x58, 0x82, 0x74, 0x51, 0x40, 0x7E, 0xEF, 0x99, 0x94, 0x7A, 0x66, 0xA6, 0x80, 0x07, 0x1B,
		0xE4, 0xED, 0x5C, 0xD4, 0x68, 0xDB, 0xBC, 0xDB, 0x12, 0x7B, 0x5B, 0xE0, 0xA7, 0x85, 0xFA, 0xD1,
		0x33, 0x61, 0x91, 0xB8, 0x37, 0x6B, 0x24, 0xC3, 0x82, 0xB5, 0x48, 0x10, 0x94, 0x32, 0x39, 0xB5,
		0x46};
#endif

/* Obfuscate key data */
#define ASP_SIGNATURE_OBFUSCATE(val) (ASP_RAND_FACTOR ^ (val)) /* Please do not modify this definition !!!! */
#endif

/* Error check  */
#if (defined ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX) && (defined ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ECDSA)
#error "You can only choose between matrix and ecdsa."
#else
#if (defined ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE) && (!defined ADVANCE_SOFTWARE_PROTECT_SIGNATURE_MATRIX) && (!defined ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ECDSA)
#error "Either matrix or ecdsa must be used."
#endif
#endif

#ifdef __cplusplus
}
#endif
#endif
