/**
 * @file    advance_software_protect_base.h
 * @brief   ICWorkShop Universal MCU Anti-Debugger library
 * @author  csh@icworkshop.com  (+86) 15989373832
 *
 * Safety License Shield
 * Copyright (c) 2017-2024, ICWorkshop Limited, All Rights Reserved
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* IMPORTANT: Please do not modify any definitions in the current file!!!! */

#ifndef __advance_software_protect_base_h
#define __advance_software_protect_base_h
#ifdef __cplusplus
extern "C"
{
#endif

/* base std c library include */
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Constant definition, please do not modify !!! */
/* compiler check */
#if defined(__IAR_SYSTEMS_ICC__)
/* IAR compiler */
#define ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR
#elif defined(__ARMCC_VERSION)
// armcc compiler (mdk)
#define ADVANCE_SOFTWARE_PROTECT_COMPILER_MDK
#elif defined(__GNUC__)
// gcc for arm compiler
#define ADVANCE_SOFTWARE_PROTECT_COMPILER_GCC
#else
// Not supported compiler!
#error "The current platform is not supported - by: cshsoft"
#endif

/* Enable Logger */
//#define ADVANCE_SOFTWARE_PROTECT_LOGGER_ENABLE

/* Enable Flash Verify */
#define ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_ENABLE

/* Anti Debugger */
//#define ADVANCE_SOFTWARE_PROTECT_ANTI_DEBUGGER_ENABLE

/* Enable Code Encrypt*/
#define ADVANCE_SOFTWARE_PROTECT_CODE_ENCRYPT_ENABLE

/* Enable Signature*/
#define ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE

/* Specifies the list of functions that need to be encrypted */
#ifdef ADVANCE_SOFTWARE_PROTECT_CODE_ENCRYPT_ENABLE
/* encryption function section */
#define ASP_ENCRYPTION_FUNCTION \
__attribute__((section(".advance_software_protect_encryption_section"))) 
/* encryption function declaration */
#if defined(ADVANCE_SOFTWARE_PROTECT_COMPILER_GCC) || \
    defined(ADVANCE_SOFTWARE_PROTECT_COMPILER_IAR)
#define ASP_ENCRYPT_F_DECL(FUNCTION_PROTOTYPE) \
    FUNCTION_PROTOTYPE ASP_ENCRYPTION_FUNCTION;
#elif defined(ADVANCE_SOFTWARE_PROTECT_COMPILER_MDK)
#define ASP_ENCRYPT_F_DECL(FUNCTION_PROTOTYPE) ASP_ENCRYPTION_FUNCTION
#endif
#else
#define ASP_ENCRYPTION_FUNCTION
#define ASP_ENCRYPT_F_DECL(FUNCTION_PROTOTYPE)
#endif

    /* Error type define */
    typedef enum E_ADV_SOFTWARE_PROTECT_ERROR_CODE
    {
        ADV_SUCCEED = 0,            /* No error */
        ADV_INVALID_PARAM,          /* Passing an illegal parameter */
        ADV_MATRIX_INNER_CFG_ERROR, /* Matrix inner config error */
        ADV_UNKNOWN_AUTH_TYPE,      /* Unknown authenticate type */
        ADV_MUTEX_USING,            /* Current is busy now */
        ADV_MEMORY_OUT_ERROR,       /* Memory error , out of memory */
        ADV_OBJECT_MODIFED,         /* Object was modifed */
        ADV_OBJECT_IS_NULL,         /* Null object */
        ADV_OBJECT_NOT_FOUND,       /* Object not exist */
        ADV_DATA_VERIFY_FAILED,     /* Data verify failed */
        ADV_SIGNATURE_TYPE_ERROR,   /* Signature config not match current api */
    } E_ADV_SOFTWARE_PROTECT_ERROR_CODE;

    /* logger */
    /* Log type*/
    #ifdef ADVANCE_SOFTWARE_PROTECT_LOGGER_ENABLE
    /* Log type*/
    typedef enum E_ADV_SOFWARE_PROTECT_LOG_TYPE
    {
        ADV_BLANK,
        ADV_INFO,
        ADV_DEBUG,
        ADV_WARN,
        ADV_EXCEPTION,
        ADV_ERROR,
        ADV_MAX
    } E_ADV_SOFWARE_PROTECT_LOG_TYPE;
    typedef void (*advance_software_logger_callback)(
        const char *msg); /* callback */
    extern bool _advance_software_protect_logger_prepare(E_ADV_SOFWARE_PROTECT_LOG_TYPE type, void **buf, int *size);
    extern bool _advance_software_protect_logger_release(void);
    #define advance_software_protect_logger(type, format, ...)                 \
    {                                                                      \
        void *_buf = 0;                                                    \
        int _size = 0;                                                     \
        if (_advance_software_protect_logger_prepare(type, &_buf, &_size)) \
        {                                                                  \
            snprintf(_buf, _size, format, ##__VA_ARGS__);                  \
            _advance_software_protect_logger_release();                    \
        }                                                                  \
    }
    #else
    #define advance_software_protect_logger(type, format, ...)            
    #endif
    /*memory callback */
    typedef void *(*calloc_func_cb)(size_t, size_t);
    typedef void (*free_func_cb)(void *);

    /* Defines wrappers for object monitoring */
    typedef struct S_ADV_SOFTWARE_PROTECT_OBJECTWATCH_INFO
    {
        bool m_auto_del;
        bool m_initialed;
        uint32_t m_hash;
        size_t m_size;
    } S_ADV_SOFTWARE_PROTECT_OBJECTWATCH_INFO;

/* Object wrapper */
#define S_ADV_SOFTWARE_PROTECT_OBJECTWATCH_Watcher(ObjectType) \
    typedef struct ObjectType##_Watcher                        \
    {                                                          \
        ObjectType m_Object;                                   \
        S_ADV_SOFTWARE_PROTECT_OBJECTWATCH_INFO m_info;        \
    } ObjectType##_Watcher;

    /* Object Watcher Exception type */
    typedef enum E_ObjectWatchException
    {
        ObjectParamError = 0,
        ObjectMemoryError,
        ObjectAlreadyExits,
        ObjectModified,
    } E_ObjectWatchException;

    /* callback for object watcher */
    typedef void (*ObjectWatcherCallback)(E_ObjectWatchException exception,
                                          void *param);

    /* Flash Verify Result */
    typedef enum E_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_RESULT
    {

        ConfigError,              /* config error ! */
        UnprocessedAfterComplied, /* Run in Debug mode */
        FlashVerifyFailed,        /* Verify Failed */
        FlashVerifyPassed         /* Verify Passed */

    } E_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_RESULT;

/* Flash Verify skiped item */
#define ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_MAGIC_S 0xACD8961B9BDD2188 /* Do not modify this definition!!! */
#define ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_MAGIC_E 0x39AD096731447C82 /* Do not modify this definition!!! */

    typedef struct S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP
    {
        void *m_skip_addr;
        uint32_t m_skip_size;
    } S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP;

/* Flash Verify Information block */
#pragma pack(push, 1)
    typedef struct S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY
    {
        uint64_t m_flash_verify_magic_s; /* magic data begin for tool */

        uint32_t m_flash_verify_skip_count;                                  /* skip flash item count */
        S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP *m_flash_verify_skip_items; /* skip flash item for some dynamic flash data ,eg sn,license,rand data*/

        uint32_t m_flash_addr_start; /* start flash addr for verify */
        uint32_t m_flash_addr_end;   /* end flash addr for verify */
        uint32_t m_total_crc32;      /* total flash checksum */

        uint64_t m_flash_verify_magic_e; /* magic data end for tool */

    } S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY;
#pragma pack(pop)

/* skip flash item verify define */
#ifdef ADVANCE_SOFTWARE_PROTECT_SIGNATURE_ENABLE
#define S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_START                                                                                                                              \
    extern const S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY m_flash_verify;                                                                                                           \
    const S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP m_flash_verify_skip_items[] = {/* skip self ,don't delete !!*/                                                              \
                                                                                  {                                                                                            \
                                                                                      .m_skip_addr = (void *)&m_flash_verify,                                                  \
                                                                                      .m_skip_size = sizeof(m_flash_verify)}, /* skip signature if enabled, don't delete !! */ \
                                                                                  {                                                                                            \
                                                                                      .m_skip_addr = (void *)&m_asp_signature_license_storage,                             \
                                                                                      .m_skip_size = sizeof(m_asp_signature_license_storage)},
#else
#define S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_START                                                                             \
    extern const S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY m_flash_verify;                                                          \
    const S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP m_flash_verify_skip_items[] = {/* skip self ,don't delete !!*/             \
                                                                                  {                                           \
                                                                                      .m_skip_addr = (void *)&m_flash_verify, \
                                                                                      .m_skip_size = sizeof(m_flash_verify)},
#endif
#define S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_END \
    }                                           \
    ;

/* Do not modify the following definitions!!! */
#define S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_APP                                                                \
    const S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY m_flash_verify = {                                               \
        .m_flash_verify_magic_s = ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_MAGIC_S,                               \
        .m_flash_verify_skip_count = sizeof(m_flash_verify_skip_items) / sizeof(m_flash_verify_skip_items[0]), \
        .m_flash_verify_skip_items = (S_ADV_SOFTWARE_PROTECT_FLASH_VERIFY_SKIP *)&m_flash_verify_skip_items,   \
        .m_flash_addr_start = UINT32_MAX,                                                                      \
        .m_flash_addr_end = 0,                                                                                 \
        .m_total_crc32 = UINT32_MAX,                                                                           \
        .m_flash_verify_magic_e = ADVANCE_SOFTWARE_PROTECT_FLASH_VERIFY_MAGIC_E};

    /* debugger event */
    typedef enum E_AntiDebuggerEvent
    {
        Development,    /* Development mode running	*/
        RDPDisabled,    /* The RDP mode is disabled	*/
        RDPEnabled,     /* The RDP mode is enabled  */
        SystemReset,    /* The system is resetting	*/
        DebuggerDisable /* Disable Debugger			*/
    } E_AntiDebuggerEvent;
    /* call back */
    typedef bool (*AntiDebuggerCallback)(E_AntiDebuggerEvent);

    /* Encrypt code Result */
    typedef enum E_ENCRYPT_CODE_EVENT
    {
        EncryptCodeUnprocessed, /* Not Processed */
        EncryptCodeOk,          /* Load Ok */
        EncryptCodeError,       /* Load Error */

    } E_ENCRYPT_CODE_EVENT;
    typedef void (*CodeLoaderCallback)(E_ENCRYPT_CODE_EVENT);

    /* Digital certificate signature */
    /* Data endian define */
    typedef enum E_DataEndian
    {
        littleEndian = 0,
        bigEndian
    } E_DataEndian;

    /* Signature type */
    /*
     *		Get Chip ID The user callback, when set to NULL, will use the
     *internal callback interface to get the chip ID, which will be useful in
     *certain situations, such as custom IDs, or when the IDs are not contiguous
     */
    typedef void (*signature_get_cid_user_cb)(uint32_t *id_addr, uint8_t size,
                                              uint32_t *id_data);

    /*
     *	Calculate the key according to the configuration
     *
     */
    typedef void (*MatrixCalc_Typedef)(char pUserKey[], char pChipID[],
                                       char pLic[]);

    typedef enum E_ADV_SOFTWARE_PROTECT_SIGNATURE_TYPE
    {
        SignatureECDSA = 0,
        SignatureMatrix,
    } E_ADV_SOFTWARE_PROTECT_SIGNATURE_TYPE;

    /* matrix */
    typedef struct E_ADV_SOFTWARE_PROTECT_MATRIX
    {

        size_t m_uid_key_length;            // The password is the same length as the user ID input
                                            // >>> Needs to be confused with m_rand_factor.
        uint32_t m_uid_key_addr;            // The matrix signature address
                                            // >>> Needs to be confused with m_rand_factor.
        uint32_t m_user_id_key[3];          // UserIDs, max 12 bytes
                                            // >>> Needs to be confused with m_rand_factor.
        E_DataEndian m_endian;              // littleEndian or bigEndian
        MatrixCalc_Typedef m_calc_callback; // Matrix calculation callbacks

    } E_ADV_SOFTWARE_PROTECT_MATRIX;

#define ECP_MAX_BITS 521 /**< Maximum bit size of groups */
#define ECDSA_MAX_SIG_LEN(bits)                        \
    (/*T,L of SEQUENCE*/ ((bits) >= 61 * 8 ? 3 : 2) +  \
     /*T,L of r,s*/ 2 * (((bits) >= 127 * 8 ? 3 : 2) + \
                         /*V of r,s*/ ((bits) + 8) / 8))
/** The maximal size of an ECDSA signature in Bytes. */
#define ECDSA_MAX_LEN ECDSA_MAX_SIG_LEN(ECP_MAX_BITS)
#define CERTIFICATION_SIG_MAX ECDSA_MAX_LEN
#pragma pack(push, 1)
    typedef struct S_ECDSA_SIGN
    {
        uint16_t m_sign_size;                       // real size
        uint8_t m_sign_data[CERTIFICATION_SIG_MAX]; // sign buffer
    } S_ECDSA_SIGN;
#pragma pack(pop)

    /* ecdsa */
    typedef struct E_ADV_SOFTWARE_PROTECT_ECDSA
    {

        const uint8_t *m_ecdsa_public_key; // ECDSA public key
        uint32_t *m_ecdsa_license_addr;    // License address
    } E_ADV_SOFTWARE_PROTECT_ECDSA;

    /* signature data */
    typedef union E_ADV_SOFTWARE_PROTECT_SIGNATURE_DATA
    {
        E_ADV_SOFTWARE_PROTECT_MATRIX m_matrix;
        E_ADV_SOFTWARE_PROTECT_ECDSA m_ecdsa;
    } E_ADV_SOFTWARE_PROTECT_SIGNATURE_DATA;

    /* signature configure object */
    typedef struct E_ADV_SOFTWARE_PROTECT_SIGNATURE
    {
        uint32_t m_rand_factor; // Digital signature verification random factor (The
                                // actual project adjusts this value)
                                // Example set as : 0x5BDB656E
                                // Signature type
        E_ADV_SOFTWARE_PROTECT_SIGNATURE_TYPE m_type;

        uint32_t m_chipid_addr; // Chip ID address location(Current Chip: STM32F103 )
                                // (Power Writer can be exported to view)
                                // Example set as : 0x1FFFF7E8
                                // >>> Needs to be confused with m_rand_factor.

        size_t m_chipid_size; // Chip ID size
                              // Example set as : 12
                              // >>> Needs to be confused with m_rand_factor.
        signature_get_cid_user_cb m_chipid_usr_cb;
        E_ADV_SOFTWARE_PROTECT_SIGNATURE_DATA m_signature_data; // signature data
    } E_ADV_SOFTWARE_PROTECT_SIGNATURE;

    /* system executed status  */
    typedef struct advance_software_protect_status
    {
        /* Signature executed */
        bool m_signature_executed;

        /* Flash verify executed */
        bool m_flash_verify_executed;

        /* Anti debugger */
        bool m_anti_debugger_executed;

    } advance_software_protect_status;

#ifdef __cplusplus
}
#endif
#endif
