/*!
    \file  cortex_chipid_binding.c
    \brief CMSIS Cortex Device Peripheral Access Layer Header File for
           Most of the Cortex Device
*/

/* Copyright (c) 2019 ICWorkShop LTD

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ICWorkShop nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE.
   ---------------------------------------------------------------------------*/
	 
/* Includes ------------------------------------------------------------------*/
#include "cortex_chipid_binding.h"

#warning Please confirm whether the UID_CHIP_ADDR&LICENCE_ADDR is correct. We suggest not to set the LICENCE_ADDR at the end of the memory
#warning MDK-ARM project->option->C/C++:C99 Mode must be selected

//Public key
const static uint8_t PUBLIC_KEY[49]={
	0x04,0x58,0x82,0x74,0x51,0x40,0x7E,0xEF,0x99,0x94,0x7A,0x66,0xA6,0x80,0x07,0x1B,
	0xE4,0xED,0x5C,0xD4,0x68,0xDB,0xBC,0xDB,0x12,0x7B,0x5B,0xE0,0xA7,0x85,0xFA,0xD1,
	0x33,0x61,0x91,0xB8,0x37,0x6B,0x24,0xC3,0x82,0xB5,0x48,0x10,0x94,0x32,0x39,0xB5,
	0x46
};



#if UID_LICENCEADDR_PLACEHOLDER_EN
//Place a placeholder for the Licence storage address
const static S_ECDSA_SIGN LicenceInfo __attribute__((at(LICENCE_ADDR^UID_CHIP_MASK)))={
#if UID_ENABLE_TAG
		UID_KEY_TAG & 0xff,(UID_KEY_TAG >> 8) & 0xff,(UID_KEY_TAG >> 16) & 0xff,(UID_KEY_TAG >> 24) & 0xff
#else
  0	
#endif
}; 
#endif

/* store full chipID */
static uint8_t mChipID[UID_CHIP_SIZE];


/* initial chip id */
static uint32_t mChipAddr = UID_CHIP_ADDR;					//global 

static bool ChipUIDAlgo(const uint8_t pChipID[],const uint16_t chipIdSize,const S_ECDSA_SIGN * licence)
{
	return sissdk_ecdsa_verify(PUBLIC_KEY,pChipID,chipIdSize,licence);
}


/**
  * @brief  Initialization chip ID. Called at initialization time
  * Note:   Please do not place ChipUIDAlgo_Calc in the same place as ChipUIDAlgo_Check
  * @retval void
  */
void ChipUIDInitial()
{
	//Do not read directly from the address,you could use your's style
	uint32_t chipIDAddr =0;
	uint32_t mask = 0x0000000f;
	int i;
	for(i = 0;i<sizeof(uint32_t) * 2;i++){
		chipIDAddr |= mChipAddr  & (mask << i *4);
	}
	//If it has been modified by decompiling, exit
	if(chipIDAddr == 0 || chipIDAddr == 0xffffffff)
		return;
	
	//get real chip ID address
	chipIDAddr^=UID_CHIP_MASK;
	//copy to array
	for(i=0;i<UID_CHIP_SIZE;i++){
		mChipID[i] = *(uint8_t *)chipIDAddr;
		chipIDAddr++;
	}

}

/**
  * @brief  Calculate the key
  * Note:   Before calling. Must have called ChipUIDInitial      
  * @retval pKey
  */
bool ChipUIDAlgo_Calc()
{
	const S_ECDSA_SIGN * pLicense;

	//Read the licence by the burner	
	uint32_t licenceAddr=LICENCE_ADDR; 
	licenceAddr^=UID_CHIP_MASK;
	
	pLicense = (S_ECDSA_SIGN *)licenceAddr;

  ChipUIDAlgo_Typedef func = ChipUIDAlgo;
	return (*func)(mChipID,UID_CHIP_SIZE,pLicense);
}

/**
  * @brief  Check the key
  * Note:   Before calling. Must have called ChipUIDInitial      
  * @retval if key is same then retun true,Otherwise, return false
  */

bool __inline ChipUIDAlgo_Check()
{
	return  ChipUIDAlgo_Calc();
}





